import { TokenManager } from "./TokenManager";
import { StickerDataEntry } from "./Interfaces/LeaderboardRelated";

export class BitmojiStickerLoader extends TokenManager {
    private stickerData: Array<StickerDataEntry> = [];

    constructor(private readonly bitmojiModule: BitmojiModule,
            private readonly remoteMediaModule: RemoteMediaModule) {
        super();
    }

    async loadForUser(snapchatUser: SnapchatUser, stickerId: string): Promise<Texture> {
        const stickerDataEntry: StickerDataEntry | undefined = this.findStickerDataEntry(snapchatUser);
        const isPromiseForStickerExists = stickerDataEntry && stickerDataEntry.stickerMap.has(stickerId);
        if (isPromiseForStickerExists) {
            return stickerDataEntry.stickerMap.get(stickerId);
        }

        const promise = this.loadBitmoji(snapchatUser, stickerId);

        if (!stickerDataEntry) {
            this.stickerData.push({
                snapchatUser: snapchatUser,
                stickerMap: new Map([[stickerId, promise]]),
            });
        } else {
            stickerDataEntry.stickerMap.set(stickerId, promise);
        }
        return promise;
    }

    private loadBitmoji(user: SnapchatUser, poseId: string): Promise<Texture> {
        const options: Bitmoji2DOptions = Bitmoji2DOptions.create();
        options.user = user;
        if (poseId !== '') {
            options.poseId = poseId;
        }

        return new Promise((resolve, reject) => {
            this.bitmojiModule.requestBitmoji2DResource(options, (resource: DynamicResource) => {
                this.remoteMediaModule.loadResourceAsImageTexture(resource,
                    texture => resolve(texture),
                    reject);
            });
        });
    }

    private findStickerDataEntry(snapchatUser: SnapchatUser): StickerDataEntry | undefined {
        return this.stickerData.find((stickerDataEntry: StickerDataEntry) => stickerDataEntry.snapchatUser.isSame(snapchatUser));
    }
}
